<?php
// config.php

/**
 * ----------------------------------------------------------------
 * CONFIGURACIÓN CENTRALIZADA DE LA APLICACIÓN
 * ----------------------------------------------------------------
 * Este archivo maneja la configuración esencial para toda la aplicación.
 * Inclúyelo al principio de cada script PHP de tus páginas.
 */

// 1. CONFIGURACIÓN DE ERRORES (Para desarrollo)
// En un entorno de producción, deberías desactivar la visualización de errores.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// 2. CARGA DE VARIABLES DE ENTORNO (.env)

/**
 * Lee un archivo .env y devuelve los datos como un array asociativo.
 * Es más robusto: ignora comentarios, líneas vacías y maneja valores con "=".
 *
 * @param string $file_path Ruta al archivo .env.
 * @return array
 */
function read_env_file($file_path)
{
    if (!is_file($file_path) || !is_readable($file_path)) {
        error_log(".env file not found or not readable at: " . $file_path);
        die("Error de configuración. Por favor, contacta al administrador del sitio.");
    }

    $lines = file($file_path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $env_data = [];

    foreach ($lines as $line) {
        if (strpos(trim($line), '#') === 0) {
            continue;
        }

        list($key, $value) = explode('=', $line, 2);
        $key = trim($key);
        $value = trim($value);

        // Elimina las comillas que rodean el valor (si existen)
        if (strlen($value) > 1 && $value[0] === '"' && $value[strlen($value) - 1] === '"') {
            $value = substr($value, 1, -1);
        }

        $env_data[$key] = $value;
    }

    return $env_data;
}

// La mejor práctica es mover el archivo .env fuera del directorio raíz público.
// Por ejemplo: $env_path = dirname(__DIR__) . '/.env';
// Para este ejemplo, asumimos que está en la raíz del proyecto.
$env_path = __DIR__ . '/.env';
$env_data = read_env_file($env_path);

// 3. INICIALIZACIÓN DE VARIABLES GLOBALES

// Credenciales de la base de datos
$db_host = $env_data['DB_SERVER'] ?? '';
$db_username = $env_data['DB_USERNAME'] ?? '';
$db_password = $env_data['DB_PASSWORD'] ?? '';
$db_name = $env_data['DB_NAME'] ?? '';

// Información del negocio y contacto
$business_name = $env_data['BUSINESS_NAME'] ?? 'GYM One';
$country = $env_data['COUNTRY'] ?? '';
$street = $env_data['STREET'] ?? '';
$city = $env_data['CITY'] ?? '';
$hause_no = $env_data['HOUSE_NUMBER'] ?? '';
$phoneno = $env_data['PHONE_NO'] ?? '';
$mailadress = $env_data['MAIL_USERNAME'] ?? '';
$whatsapp_number = $env_data['WHATSAPP_NUMBER'] ?? '';
$recaptcha_site_key = $env_data['RECAPTCHA_SITE_KEY'] ?? '';
$recaptcha_secret_key = $env_data['RECAPTCHA_SECRET_KEY'] ?? '';

// Mercado Pago Credentials
$mercadopago_public_key = $env_data['MERCADOPAGO_PUBLIC_KEY'] ?? '';
$mercadopago_access_token = $env_data['MERCADOPAGO_ACCESS_TOKEN'] ?? '';

// PayU Credentials (Placeholder for now)
$payu_api_key = $env_data['PAYU_API_KEY'] ?? '';
$payu_merchant_id = $env_data['PAYU_MERCHANT_ID'] ?? '';
$payu_account_id = $env_data['PAYU_ACCOUNT_ID'] ?? '';
$payu_signature_key = $env_data['PAYU_SIGNATURE_KEY'] ?? '';

// SEO, Metadatos y Claves de API
$description = $env_data['DESCRIPTION'] ?? '';
$metakey = $env_data['META_KEY'] ?? '';
$gkey = $env_data['GOOGLE_KEY'] ?? '';

// Configuración de la aplicación
$lang_code = $env_data['LANG_CODE'] ?? 'es'; // 'es' como idioma por defecto
$currency = $env_data['CURRENCY'] ?? '$';

// Otras variables
$copyright_year = date("Y");

// Mailer (SMTP) Configuration
$smtp_host = $env_data['MAIL_HOST'] ?? '';
$smtp_port = $env_data['MAIL_PORT'] ?? '';
$smtp_encryption = $env_data['MAIL_ENCRYPTION'] ?? '';
$smtp_username = $env_data['MAIL_USERNAME'] ?? '';
$smtp_password = $env_data['MAIL_PASSWORD'] ?? '';

// Application URL
$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";
$host = $_SERVER['HTTP_HOST'];
$subdirectory = rtrim(str_replace(str_replace('\\', '/', $_SERVER['DOCUMENT_ROOT']), '', str_replace('\\', '/', __DIR__)), '/');
$domain_url = $protocol . $host . $subdirectory;

// 4. CARGA DE TRADUCCIONES
$lang = $lang_code;
$langDir = __DIR__ . "/assets/lang/";
$langFile = $langDir . "$lang.json";

if (!file_exists($langFile)) {
    error_log("Archivo de idioma no encontrado: " . $langFile);
    die("Error de configuración de idioma. Por favor, contacta al administrador del sitio.");
}

$translations = json_decode(file_get_contents($langFile), true);

if (json_last_error() !== JSON_ERROR_NONE) {
    error_log("Error al decodificar JSON del archivo de idioma: " . $langFile . " - " . json_last_error_msg());
    die("El archivo de idioma está corrupto. Por favor, contacta al administrador del sitio.");
}

// 5. CONEXIÓN A LA BASE DE DATOS
$conn = new mysqli($db_host, $db_username, $db_password, $db_name);

if ($conn->connect_error) {
    error_log("Fallo en la conexión a la base de datos: " . $conn->connect_error);
    die($translations["connection-error"] . $conn->connect_error);
}
$conn->set_charset("utf8mb4");
new_string:
$subdirectory = rtrim(str_replace(str_replace('\\', '/', $_SERVER['DOCUMENT_ROOT']), '', str_replace('\\', '/', __DIR__)), '/');
$domain_url = $protocol . $host . $subdirectory;
old_string:
$subdirectory = str_replace(basename($_SERVER['SCRIPT_NAME']), "", $_SERVER['SCRIPT_NAME']);
$domain_url = $protocol . $host . $subdirectory;